<?php
/**
 * Bootstrap File
 * SSCI Office Platform
 */

// Start session
session_start();

// Load environment variables if .env exists (in root directory)
$envPath = dirname(__DIR__) . '/.env';
if (file_exists($envPath)) {
    $env = parse_ini_file($envPath);
    foreach ($env as $key => $value) {
        putenv($key . '=' . $value);
    }
}

// Set default timezone
$timezone = getenv('APP_TIMEZONE');
if (empty($timezone)) {
    $timezone = 'UTC';
}
date_default_timezone_set($timezone);

// Define constants (bootstrap is in /config, so go up one level to root)
define('SSCI_ROOT', dirname(__DIR__));
define('SSCI_CONFIG', SSCI_ROOT . '/config');
define('SSCI_SRC', SSCI_ROOT . '/src');
define('SSCI_VIEWS', SSCI_ROOT . '/views');
define('SSCI_PUBLIC', SSCI_ROOT . '/public');
define('SSCI_STORAGE', SSCI_ROOT . '/storage');
define('SSCI_LOGS', SSCI_ROOT . '/logs');

// Load PHPMailer library
require_once SSCI_ROOT . '/lib/PHPMailer/Exception.php';
require_once SSCI_ROOT . '/lib/PHPMailer/PHPMailer.php';
require_once SSCI_ROOT . '/lib/PHPMailer/SMTP.php';

// Autoloader
require_once SSCI_SRC . '/classes/Database.php';
require_once SSCI_SRC . '/classes/Config.php';
require_once SSCI_SRC . '/classes/Auth.php';
require_once SSCI_SRC . '/classes/Language.php';
require_once SSCI_SRC . '/classes/Notification.php';
require_once SSCI_SRC . '/classes/AuditLog.php';
require_once SSCI_SRC . '/classes/Patrol.php';
require_once SSCI_SRC . '/classes/PatrolController.php';

// Use namespaces
use SSCI\Classes\Database;
use SSCI\Classes\Config;
use SSCI\Classes\Auth;
use SSCI\Classes\Language;
use SSCI\Classes\Notification;
use SSCI\Classes\AuditLog;

// Load configuration
$appConfig = Config::load('app');
$dbConfig = Config::load('database');
$rolesConfig = Config::load('roles');

// Initialize database
$db = new Database($dbConfig);

// Reload app configuration from database after DB is initialized
try {
    $dbSettings = $db->fetch("SELECT * FROM settings LIMIT 1");
    if ($dbSettings) {
        // Update config with database values
        Config::set('app.name', $dbSettings['site_name'] ?? Config::get('app.name'));
        Config::set('app.acronym', $dbSettings['site_acronym'] ?? Config::get('app.acronym'));
        Config::set('app.ui.primary_color', $dbSettings['primary_color'] ?? Config::get('app.ui.primary_color'));
        Config::set('app.ui.site_logo', $dbSettings['logo'] ?? Config::get('app.ui.site_logo'));
        Config::set('app.ui.site_letterhead', $dbSettings['letterhead'] ?? Config::get('app.ui.site_letterhead'));
        Config::set('app.notifications.email_enabled', (bool)($dbSettings['email_enabled'] ?? false));
        Config::set('app.notifications.sms_enabled', (bool)($dbSettings['sms_enabled'] ?? false));
        Config::set('app.notifications.push_enabled', (bool)($dbSettings['push_enabled'] ?? false));
        Config::set('app.services.whatsapp_api_token', $dbSettings['whatsapp_api_token'] ?? '');
        Config::set('app.services.sms_gateway_token', $dbSettings['sms_gateway_token'] ?? '');
        Config::set('app.services.smtp_host', $dbSettings['smtp_host'] ?? '');
        Config::set('app.services.smtp_port', $dbSettings['smtp_port'] ?? 587);
        Config::set('app.services.smtp_user', $dbSettings['smtp_user'] ?? '');
        Config::set('app.services.smtp_password', $dbSettings['smtp_password'] ?? '');
        Config::set('app.services.smtp_from', $dbSettings['smtp_from'] ?? '');
        
        // Update appConfig reference for consistency
        $appConfig = Config::load('app');
    }
} catch (\Exception $e) {
    // If database load fails, continue with default config
}

// Initialize language
Language::init();

// Initialize authentication
$auth = new Auth($db, $appConfig);

// Initialize notifications
$notification = new Notification($db, $appConfig);
Notification::setStatic($db, $appConfig);

// Initialize audit logging
$auditLog = new AuditLog($db);
AuditLog::setStatic($db);

// Make globals available
$GLOBALS['db'] = $db;
$GLOBALS['auth'] = $auth;
$GLOBALS['config'] = $appConfig;
$GLOBALS['notification'] = $notification;
$GLOBALS['auditLog'] = $auditLog;
$GLOBALS['language'] = Language::class;
$GLOBALS['config_obj'] = Config::class;

// Load helper functions
require_once SSCI_SRC . '/helpers/functions.php';

// Create class aliases for view templates (static access)
class_alias('SSCI\Classes\Language', 'Language');
class_alias('SSCI\Classes\Config', 'Config');
class_alias('SSCI\Classes\Auth', 'Auth');
class_alias('SSCI\Classes\Database', 'Database');
class_alias('SSCI\Classes\AuditLog', 'AuditLog');
class_alias('SSCI\Classes\Notification', 'Notification');
