/**
 * Notification Handler
 * SSCI Office Platform
 */

class NotificationHandler {
    constructor() {
        this.notifications = [];
    }

    /**
     * Show in-app notification
     */
    showNotification(title, message, type = 'info', duration = 3000) {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type} animate__animated animate__slideInUp`;
        notification.innerHTML = `
            <div style="padding: 16px; background: white; border-radius: 8px; box-shadow: 0 4px 6px rgba(0,0,0,0.1); border-left: 4px solid ${this.getColor(type)};">
                <div style="display: flex; align-items: flex-start; gap: 12px;">
                    <i class="fas ${this.getIcon(type)}" style="color: ${this.getColor(type)}; margin-top: 4px;"></i>
                    <div>
                        <p style="font-weight: bold; margin: 0; color: #1f2937;">${title}</p>
                        <p style="margin: 4px 0 0 0; color: #6b7280; font-size: 14px;">${message}</p>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(notification);

        if (duration > 0) {
            setTimeout(() => {
                notification.classList.add('animate__fadeOut');
                setTimeout(() => notification.remove(), 500);
            }, duration);
        }

        return notification;
    }

    /**
     * Get icon for notification type
     */
    getIcon(type) {
        const icons = {
            success: 'fa-check-circle',
            error: 'fa-exclamation-circle',
            warning: 'fa-warning',
            info: 'fa-info-circle'
        };
        return icons[type] || icons.info;
    }

    /**
     * Get color for notification type
     */
    getColor(type) {
        const colors = {
            success: '#10b981',
            error: '#ef4444',
            warning: '#f59e0b',
            info: '#2563eb'
        };
        return colors[type] || colors.info;
    }

    /**
     * Send push notification
     */
    sendPushNotification(title, message, options = {}) {
        if (!('Notification' in window)) {
            console.log('This browser does not support notifications');
            return;
        }

        if (Notification.permission === 'granted') {
            new Notification(title, {
                icon: '/images/logo.png',
                badge: '/images/badge.png',
                body: message,
                ...options
            });
        } else if (Notification.permission !== 'denied') {
            Notification.requestPermission().then(permission => {
                if (permission === 'granted') {
                    new Notification(title, {
                        icon: '/images/logo.png',
                        badge: '/images/badge.png',
                        body: message,
                        ...options
                    });
                }
            });
        }
    }

    /**
     * Request notification permission
     */
    requestPermission() {
        if ('Notification' in window && Notification.permission === 'default') {
            Notification.requestPermission();
        }
    }
}

// Initialize notification handler
const notifier = new NotificationHandler();

// Request notification permission on load
window.addEventListener('load', () => {
    notifier.requestPermission();
});
