<?php
/**
 * Settings Management Page
 * SSCI Office Platform
 */

$auth = $GLOBALS['auth'];
$db = $GLOBALS['db'];
$pageTitle = trans('settings') . ' - ' . config('app.name');

if (!$auth->canAccessPage('settings')) {
    header('Location: ?page=dashboard');
    exit();
}

require_once __DIR__ . '/../../layouts/header.php';

// Load settings from database
$settings = $db->fetch("SELECT * FROM settings LIMIT 1") ?? [
    'site_name'     => config('app.name'),
    'site_acronym'  => config('app.acronym'),
    'primary_color' => '#2563eb',
];

?>
<script>
/* ============================================================
   SweetAlert Notification Helper
============================================================ */
function notify(type, message) {
    Swal.fire({
        icon: type,
        title: message,
        toast: true,
        position: 'top-end',
        timer: 2200,
        showConfirmButton: false
    });
}

/* ============================================================
   Tab Switching
============================================================ */
function switchTab(tab) {
    document.querySelectorAll('.tab-content').forEach(t => t.classList.add('hidden'));
    document.getElementById(tab + "-tab").classList.remove('hidden');

    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('text-blue-600', 'border-b-2', 'border-blue-600');
        btn.classList.add('text-gray-600');
    });

    const active = document.querySelector(`[data-tab="${tab}"]`);
    if (active) {
        active.classList.remove('text-gray-600');
        active.classList.add('text-blue-600', 'border-b-2', 'border-blue-600');
    }
}

/* ============================================================
   Generate App Key
============================================================ */
function generateAppKey() {
    let key = "base64:" + btoa(crypto.getRandomValues(new Uint8Array(12)).toString());
    const input = document.querySelector('input[name="app_key"]');
    if (input) input.value = key;
    notify('success', "<?php echo trans('key_generated'); ?>");
}

/* ============================================================
   Backup Database
============================================================ */
function backupDatabase() {
    Swal.fire({
        title: "<?php echo trans('backup_database'); ?>",
        text: "<?php echo trans('backup_in_progress'); ?>",
        icon: "info",
        didOpen: () => Swal.showLoading()
    });
}

/* ============================================================
   Clear Cache
============================================================ */
function clearCache() {
    Swal.fire({
        title: "<?php echo trans('clear_cache'); ?>",
        text: "<?php echo trans('confirm_clear_cache'); ?>",
        icon: "warning",
        showCancelButton: true,
    }).then(result => {
        if (result.isConfirmed) {
            notify("success", "<?php echo trans('cache_cleared'); ?>");
        }
    });
}

/* ============================================================
   Universal AJAX Form Handler
============================================================ */
document.addEventListener("DOMContentLoaded", () => {
    const formActions = {
        generalSettingsForm:       "saveGeneralSettings",
        appearanceSettingsForm:    "saveAppearanceSettings",
        emailSettingsForm:         "saveEmailSettings",
        notificationSettingsForm:  "saveNotificationSettings",
        appSettingsForm:           "saveAppSettings",
        databaseSettingsForm:      "saveDatabaseSettings",
        paymentGatewayForm:        "savePaymentGatewaySettings"
    };

    Object.entries(formActions).forEach(([formId, action]) => {
        const form = document.getElementById(formId);
        if (!form) return;

        form.addEventListener("submit", async e => {
            e.preventDefault();
            e.stopPropagation();

            let btn = form.querySelector('button[type="submit"]');
            let original = btn.innerHTML;

            btn.disabled = true;
            btn.innerHTML = `<i class="fas fa-spinner fa-spin mr-2"></i><?php echo trans("saving"); ?>`;

            try {
                const res = await fetch(`?page=api&action=${action}`, {
                    method: "POST",
                    body: new FormData(form)
                });

                const json = await res.json();

                if (json.success) {
                    notify("success", json.message || "<?php echo trans('settings_updated_successfully'); ?>");
                    setTimeout(() => location.reload(), 300);
                } else {
                    notify("error", json.message || "<?php echo trans('an_error_occurred'); ?>");
                }

            } catch (err) {
                console.error(err);
                notify("error", "<?php echo trans('an_error_occurred'); ?>");
            }

            btn.disabled = false;
            btn.innerHTML = original;
        });
    });
});
</script>

<!-- PAGE HEADER -->
<div class="animate__animated animate__fadeIn w-full mx-auto px-4 sm:px-6 lg:px-8 max-w-4xl">
    <div class="mb-6">
        <h1 class="text-2xl sm:text-3xl font-bold text-gray-800">
            <i class="fas fa-cog text-blue-600 mr-3"></i><?php echo trans('settings'); ?>
        </h1>
        <p class="text-gray-600 text-sm mt-1"><?php echo trans('manage_site_settings') ?? 'Configure your site settings and preferences'; ?></p>
    </div>

    <!-- TABS NAVIGATION -->
    <div class="bg-white rounded-lg shadow overflow-hidden mb-6">
        <div class="flex border-b border-gray-200 overflow-x-auto">
            <button onclick="switchTab('general')" data-tab="general" class="flex-1 min-w-max px-4 sm:px-6 py-4 text-center font-semibold tab-btn text-blue-600 border-b-2 border-blue-600 transition-colors">
                <i class="fas fa-sliders-h mr-2"></i><span class="hidden sm:inline"><?php echo trans('general') ?? 'General'; ?></span><span class="sm:hidden text-xs">General</span>
            </button>
            <button onclick="switchTab('appearance')" data-tab="appearance" class="flex-1 min-w-max px-4 sm:px-6 py-4 text-center font-semibold tab-btn text-gray-600 transition-colors">
                <i class="fas fa-palette mr-2"></i><span class="hidden sm:inline"><?php echo trans('appearance') ?? 'Appearance'; ?></span><span class="sm:hidden text-xs">Appearance</span>
            </button>
            <button onclick="switchTab('email')" data-tab="email" class="flex-1 min-w-max px-4 sm:px-6 py-4 text-center font-semibold tab-btn text-gray-600 transition-colors">
                <i class="fas fa-envelope mr-2"></i><span class="hidden sm:inline"><?php echo trans('email_settings') ?? 'Email'; ?></span><span class="sm:hidden text-xs">Email</span>
            </button>
            <button onclick="switchTab('notifications')" data-tab="notifications" class="flex-1 min-w-max px-4 sm:px-6 py-4 text-center font-semibold tab-btn text-gray-600 transition-colors">
                <i class="fas fa-bell mr-2"></i><span class="hidden sm:inline"><?php echo trans('notifications'); ?></span><span class="sm:hidden text-xs">Notify</span>
            </button>
            <button onclick="switchTab('advanced')" data-tab="advanced" class="flex-1 min-w-max px-4 sm:px-6 py-4 text-center font-semibold tab-btn text-gray-600 transition-colors">
                <i class="fas fa-wrench mr-2"></i><span class="hidden sm:inline"><?php echo trans('advanced') ?? 'Advanced'; ?></span><span class="sm:hidden text-xs">Advanced</span>
            </button>
        </div>
    </div>

    <!-- GENERAL SETTINGS TAB -->
    <div id="general-tab" class="tab-content">
        <form id="generalSettingsForm" class="bg-white rounded-lg shadow p-6 sm:p-8 space-y-6">
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800 pb-4 border-b-2 border-blue-200">
                <i class="fas fa-building text-blue-600 mr-3"></i><?php echo trans('general_settings') ?? 'General Settings'; ?>
            </h2>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <!-- Site Name -->
                <div>
                    <label class="block text-gray-700 font-semibold mb-2">
                        <i class="fas fa-heading text-blue-600 mr-2"></i><?php echo trans('site_name'); ?>
                    </label>
                    <input type="text" name="site_name" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                           value="<?php echo htmlspecialchars($settings['site_name'] ?? ''); ?>" placeholder="SSCI Office">
                </div>

                <!-- Site Acronym -->
                <div>
                    <label class="block text-gray-700 font-semibold mb-2">
                        <i class="fas fa-abbreviation text-blue-600 mr-2"></i><?php echo trans('site_acronym'); ?>
                    </label>
                    <input type="text" name="site_acronym" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                           value="<?php echo htmlspecialchars($settings['site_acronym'] ?? ''); ?>" placeholder="SSCI" maxlength="10">
                </div>

                <!-- Email -->
                <div>
                    <label class="block text-gray-700 font-semibold mb-2">
                        <i class="fas fa-envelope text-blue-600 mr-2"></i><?php echo trans('email'); ?>
                    </label>
                    <input type="email" name="email" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                           value="<?php echo htmlspecialchars($settings['email'] ?? ''); ?>" placeholder="info@ssci.local">
                </div>

                <!-- Phone -->
                <div>
                    <label class="block text-gray-700 font-semibold mb-2">
                        <i class="fas fa-phone text-blue-600 mr-2"></i><?php echo trans('phone'); ?>
                    </label>
                    <input type="tel" name="phone" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                           value="<?php echo htmlspecialchars($settings['phone'] ?? ''); ?>" placeholder="+254 712 345 678">
                </div>
            </div>

            <!-- Address -->
            <div>
                <label class="block text-gray-700 font-semibold mb-2">
                    <i class="fas fa-map-pin text-blue-600 mr-2"></i><?php echo trans('address') ?? 'Address'; ?>
                </label>
                <textarea name="address" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                          rows="3" placeholder="Office address"><?php echo htmlspecialchars($settings['address'] ?? ''); ?></textarea>
            </div>

            <!-- Save Button -->
            <div class="flex gap-3 pt-4 border-t border-gray-200">
                <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition font-semibold flex items-center">
                    <i class="fas fa-save mr-2"></i><?php echo trans('save_changes') ?? 'Save Changes'; ?>
                </button>
            </div>
        </form>
    </div>

    <!-- APPEARANCE SETTINGS TAB -->
    <div id="appearance-tab" class="tab-content hidden">
        <form id="appearanceSettingsForm" class="bg-white rounded-lg shadow p-6 sm:p-8 space-y-6">
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800 pb-4 border-b-2 border-blue-200">
                <i class="fas fa-palette text-blue-600 mr-3"></i><?php echo trans('appearance_settings') ?? 'Appearance Settings'; ?>
            </h2>

            <!-- Primary Color -->
            <div>
                <label class="block text-gray-700 font-semibold mb-2">
                    <i class="fas fa-droplet text-blue-600 mr-2"></i><?php echo trans('primary_color'); ?>
                </label>
                <div class="flex flex-col sm:flex-row items-start sm:items-center gap-4">
                    <input type="color" name="primary_color" class="h-16 w-20 cursor-pointer border-2 border-gray-300 rounded-lg" 
                           value="<?php echo htmlspecialchars($settings['primary_color'] ?? '#2563eb'); ?>">
                    <input type="text" name="primary_color_hex" class="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500" 
                           value="<?php echo htmlspecialchars($settings['primary_color'] ?? '#2563eb'); ?>" placeholder="#2563eb">
                </div>
            </div>

            <!-- Logo Upload -->
            <div>
                <label class="block text-gray-700 font-semibold mb-2">
                    <i class="fas fa-image text-blue-600 mr-2"></i><?php echo trans('site_logo'); ?>
                </label>
                <div class="flex flex-col sm:flex-row items-start sm:items-center gap-4">
                    <?php if (isset($settings['logo']) && $settings['logo']): ?>
                        <img src="<?php echo htmlspecialchars($settings['logo']); ?>" alt="Logo" class="h-16 w-16 rounded-lg object-cover border border-gray-300">
                    <?php else: ?>
                        <div class="h-16 w-16 rounded-lg bg-gray-200 flex items-center justify-center flex-shrink-0">
                            <i class="fas fa-image text-gray-400 text-2xl"></i>
                        </div>
                    <?php endif; ?>
                    <input type="file" name="logo" class="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:outline-none" accept="image/*">
                </div>
            </div>

            <!-- Letterhead Upload -->
            <div>
                <label class="block text-gray-700 font-semibold mb-2">
                    <i class="fas fa-file-image text-blue-600 mr-2"></i><?php echo trans('site_letterhead') ?? 'Letterhead'; ?>
                </label>
                <input type="file" name="letterhead" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none" accept="image/*">
            </div>

            <!-- Save Button -->
            <div class="flex gap-3 pt-4 border-t border-gray-200">
                <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition font-semibold flex items-center">
                    <i class="fas fa-save mr-2"></i><?php echo trans('save_changes') ?? 'Save Changes'; ?>
                </button>
            </div>
        </form>
    </div>

    <!-- EMAIL SETTINGS TAB -->
    <div id="email-tab" class="tab-content hidden">
        <form id="emailSettingsForm" class="bg-white rounded-lg shadow p-6 sm:p-8 space-y-6">
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800 pb-4 border-b-2 border-blue-200">
                <i class="fas fa-envelope text-blue-600 mr-3"></i><?php echo trans('email_settings') ?? 'Email Settings'; ?>
            </h2>

            <!-- Enable Email -->
            <div>
                <label class="flex items-center gap-3 cursor-pointer">
                    <input type="checkbox" name="email_enabled" class="w-5 h-5 rounded border-gray-300" <?php echo ($settings['email_enabled'] ?? 0) ? 'checked' : ''; ?>>
                    <span class="text-gray-700 font-semibold"><?php echo trans('enable_email') ?? 'Enable Email Notifications'; ?></span>
                </label>
            </div>

            <!-- SMTP Settings -->
            <div class="border-t pt-6">
                <h3 class="text-lg font-semibold text-gray-700 mb-4 flex items-center">
                    <i class="fas fa-server text-blue-600 mr-2"></i><?php echo trans('smtp_settings') ?? 'SMTP Settings'; ?>
                </h3>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <!-- SMTP Host -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-server text-blue-600 mr-2"></i><?php echo trans('smtp_host') ?? 'SMTP Host'; ?>
                        </label>
                        <input type="text" name="smtp_host" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['smtp_host'] ?? ''); ?>" placeholder="smtp.gmail.com">
                    </div>

                    <!-- SMTP Port -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-plug text-blue-600 mr-2"></i><?php echo trans('smtp_port') ?? 'SMTP Port'; ?>
                        </label>
                        <input type="number" name="smtp_port" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['smtp_port'] ?? '587'); ?>" placeholder="587">
                    </div>

                    <!-- SMTP User -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-user text-blue-600 mr-2"></i><?php echo trans('smtp_user') ?? 'SMTP Username'; ?>
                        </label>
                        <input type="email" name="smtp_user" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['smtp_user'] ?? ''); ?>" placeholder="your-email@gmail.com">
                    </div>

                    <!-- SMTP Password -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-key text-blue-600 mr-2"></i><?php echo trans('smtp_password') ?? 'SMTP Password'; ?>
                        </label>
                        <input type="password" name="smtp_password" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['smtp_password'] ?? ''); ?>" placeholder="Enter SMTP password">
                    </div>

                    <!-- From Email -->
                    <div class="md:col-span-2">
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-envelope text-blue-600 mr-2"></i><?php echo trans('from_email') ?? 'From Email Address'; ?>
                        </label>
                        <input type="email" name="smtp_from" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['smtp_from'] ?? ''); ?>" placeholder="noreply@ssci.local">
                    </div>
                </div>
            </div>

            <!-- Save Button -->
            <div class="flex gap-3 pt-4 border-t border-gray-200">
                <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition font-semibold flex items-center">
                    <i class="fas fa-save mr-2"></i><?php echo trans('save_changes') ?? 'Save Changes'; ?>
                </button>
            </div>
        </form>
    </div>
    <!-- NOTIFICATIONS SETTINGS TAB -->
    <div id="notifications-tab" class="tab-content hidden">
        <form id="notificationSettingsForm" class="bg-white rounded-lg shadow p-6 sm:p-8 space-y-6">
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800 pb-4 border-b-2 border-blue-200">
                <i class="fas fa-bell text-blue-600 mr-3"></i><?php echo trans('notification_settings'); ?>
            </h2>

            <!-- SMS Settings -->
            <div class="border-b pb-6">
                <h3 class="text-lg font-semibold text-gray-700 mb-4 flex items-center">
                    <i class="fas fa-comment text-blue-600 mr-2"></i><?php echo trans('sms_settings') ?? 'SMS Settings'; ?>
                </h3>
                <div class="space-y-4">
                    <label class="flex items-center gap-3 cursor-pointer">
                        <input type="checkbox" name="sms_notifications_enabled" class="w-5 h-5 rounded border-gray-300" <?php echo ($settings['sms_notifications_enabled'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="text-gray-700 font-semibold"><?php echo trans('enable_sms_notifications') ?? 'Enable SMS Notifications'; ?></span>
                    </label>
                    
                    <!-- SMS Username -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-user text-blue-600 mr-2"></i><?php echo trans('sms_username') ?? 'SMS Gateway Username'; ?>
                        </label>
                        <input type="text" name="sms_username" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['sms_username'] ?? ''); ?>" placeholder="Enter SMS gateway username (sms-gate.app)">
                    </div>
                    
                    <!-- SMS Password -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-lock text-blue-600 mr-2"></i><?php echo trans('sms_password') ?? 'SMS Gateway Password'; ?>
                        </label>
                        <input type="password" name="sms_password" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['sms_password'] ?? ''); ?>" placeholder="Enter SMS gateway password">
                    </div>
                </div>
            </div>

            <!-- WhatsApp Settings -->
            <div class="border-b pb-6">
                <h3 class="text-lg font-semibold text-gray-700 mb-4 flex items-center">
                    <i class="fab fa-whatsapp text-green-600 mr-2"></i><?php echo trans('whatsapp_settings') ?? 'WhatsApp Settings'; ?>
                </h3>
                <div class="space-y-4">
                    <label class="flex items-center gap-3 cursor-pointer">
                        <input type="checkbox" name="whatsapp_notifications_enabled" class="w-5 h-5 rounded border-gray-300" <?php echo ($settings['whatsapp_notifications_enabled'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="text-gray-700 font-semibold"><?php echo trans('enable_whatsapp_notifications') ?? 'Enable WhatsApp Notifications'; ?></span>
                    </label>
                    
                    <!-- WhatsApp Instance ID -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-hashtag text-green-600 mr-2"></i><?php echo trans('whatsapp_instance_id') ?? 'WhatsApp Instance ID'; ?>
                        </label>
                        <input type="text" name="whatsapp_instance_id" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['whatsapp_instance_id'] ?? ''); ?>" placeholder="E.g., 100571 (from ultramsg.com)">
                        <p class="text-xs text-gray-500 mt-1"><?php echo trans('whatsapp_instance_id_help') ?? 'Get your instance ID from ultramsg.com dashboard'; ?></p>
                    </div>
                    
                    <!-- WhatsApp API Token -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-key text-green-600 mr-2"></i><?php echo trans('whatsapp_api_token') ?? 'WhatsApp API Token'; ?>
                        </label>
                        <input type="password" name="whatsapp_api_token" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['whatsapp_api_token'] ?? ''); ?>" placeholder="Enter your WhatsApp API token from ultramsg.com">
                    </div>
                </div>
            </div>

            <!-- Push Notifications -->
            <div>
                <h3 class="text-lg font-semibold text-gray-700 mb-4 flex items-center">
                    <i class="fas fa-bell text-blue-600 mr-2"></i><?php echo trans('push_notifications') ?? 'Push Notifications'; ?>
                </h3>
                <div class="space-y-4">
                    <label class="flex items-center gap-3 cursor-pointer">
                        <input type="checkbox" name="push_enabled" class="w-5 h-5 rounded border-gray-300" <?php echo ($settings['push_enabled'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="text-gray-700 font-semibold"><?php echo trans('enable_push_notifications') ?? 'Enable Push Notifications'; ?></span>
                    </label>
                    
                    <!-- VAPID Public Key -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-key text-blue-600 mr-2"></i><?php echo trans('vapid_public_key') ?? 'VAPID Public Key'; ?>
                        </label>
                        <textarea name="vapid_public_key" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition font-mono text-xs" 
                                  rows="3" placeholder="Enter your VAPID public key"><?php echo htmlspecialchars($settings['vapid_public_key'] ?? ''); ?></textarea>
                    </div>
                    
                    <!-- VAPID Private Key -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2">
                            <i class="fas fa-key text-blue-600 mr-2"></i><?php echo trans('vapid_private_key') ?? 'VAPID Private Key'; ?>
                        </label>
                        <textarea name="vapid_private_key" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition font-mono text-xs" 
                                  rows="3" placeholder="Enter your VAPID private key"><?php echo htmlspecialchars($settings['vapid_private_key'] ?? ''); ?></textarea>
                    </div>
                </div>
            </div>

            <!-- Save Button -->
            <div class="flex gap-3 pt-4 border-t border-gray-200">
                <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition font-semibold flex items-center">
                    <i class="fas fa-save mr-2"></i><?php echo trans('save_changes') ?? 'Save Changes'; ?>
                </button>
            </div>
        </form>
    </div>

    <!-- ADVANCED SETTINGS TAB -->
    <div id="advanced-tab" class="tab-content hidden">
        <!-- Application Settings -->
        <form id="appSettingsForm" class="bg-white rounded-lg shadow p-6 sm:p-8 space-y-6 mb-6">
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800 pb-4 border-b-2 border-blue-200">
                <i class="fas fa-cogs text-blue-600 mr-3"></i><?php echo trans('app_settings') ?? 'Application Settings'; ?>
            </h2>
            <div class="space-y-4">
                <!-- App Debug -->
                <label class="flex items-center gap-3 cursor-pointer">
                    <input type="checkbox" name="app_debug" class="w-5 h-5 rounded border-gray-300" <?php echo ($settings['app_debug'] ?? 0) ? 'checked' : ''; ?>>
                    <span class="text-gray-700 font-semibold"><?php echo trans('enable_debug_mode') ?? 'Enable Debug Mode'; ?></span>
                </label>

                <!-- App URL -->
                <div>
                    <label class="block text-gray-700 font-semibold mb-2"><?php echo trans('app_url') ?? 'Application URL'; ?></label>
                    <input type="url" name="app_url" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                           value="<?php echo htmlspecialchars($settings['app_url'] ?? ''); ?>" placeholder="https://portal.example.com">
                </div>

                <!-- App Key -->
                <div>
                    <label class="block text-gray-700 font-semibold mb-2"><?php echo trans('app_key') ?? 'Application Key'; ?></label>
                    <div class="flex gap-2">
                        <input type="password" name="app_key" class="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['app_key'] ?? ''); ?>" placeholder="Enter application key">
                        <button type="button" onclick="generateAppKey()" class="bg-blue-600 text-white px-4 py-3 rounded-lg hover:bg-blue-700 transition font-semibold">
                            <i class="fas fa-sync mr-2"></i><?php echo trans('generate'); ?>
                        </button>
                    </div>
                    <p class="text-xs text-gray-500 mt-1"><?php echo trans('app_key_description') ?? 'This is a secret key used for encryption'; ?></p>
                </div>

                <!-- Save Button -->
                <div class="flex gap-3 pt-4 border-t border-gray-200">
                    <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition font-semibold flex items-center">
                        <i class="fas fa-save mr-2"></i><?php echo trans('save_changes') ?? 'Save Changes'; ?>
                    </button>
                </div>
            </div>
        </form>

        <!-- Database Settings -->
        <form id="databaseSettingsForm" class="bg-white rounded-lg shadow p-6 sm:p-8 space-y-6 mb-6">
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800 pb-4 border-b-2 border-blue-200">
                <i class="fas fa-database text-blue-600 mr-3"></i><?php echo trans('database_settings') ?? 'Database Settings'; ?>
            </h2>
            <div class="space-y-4">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <!-- DB Host -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2"><?php echo trans('db_host') ?? 'Database Host'; ?></label>
                        <input type="text" name="db_host" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['db_host'] ?? ''); ?>" placeholder="localhost">
                    </div>

                    <!-- DB Port -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2"><?php echo trans('db_port') ?? 'Database Port'; ?></label>
                        <input type="number" name="db_port" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['db_port'] ?? '3306'); ?>" placeholder="3306">
                    </div>

                    <!-- DB Name -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2"><?php echo trans('db_name') ?? 'Database Name'; ?></label>
                        <input type="text" name="db_name" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['db_name'] ?? ''); ?>" placeholder="database_name">
                    </div>

                    <!-- DB User -->
                    <div>
                        <label class="block text-gray-700 font-semibold mb-2"><?php echo trans('db_user') ?? 'Database User'; ?></label>
                        <input type="text" name="db_user" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['db_user'] ?? ''); ?>" placeholder="database_user">
                    </div>

                    <!-- DB Password -->
                    <div class="md:col-span-2">
                        <label class="block text-gray-700 font-semibold mb-2"><?php echo trans('db_password') ?? 'Database Password'; ?></label>
                        <input type="password" name="db_password" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 focus:ring-2 focus:ring-blue-200 transition" 
                               value="<?php echo htmlspecialchars($settings['db_password'] ?? ''); ?>" placeholder="Enter database password">
                        <p class="text-xs text-gray-500 mt-1"><?php echo trans('leave_blank_to_keep_current') ?? 'Leave blank to keep current password'; ?></p>
                    </div>
                </div>

                <!-- Save Button -->
                <div class="flex gap-3 pt-4 border-t border-gray-200">
                    <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition font-semibold flex items-center">
                        <i class="fas fa-save mr-2"></i><?php echo trans('save_changes') ?? 'Save Changes'; ?>
                    </button>
                </div>
            </div>
        </form>

        <!-- Payment Gateway Settings -->
        <form id="paymentGatewayForm" class="bg-white rounded-lg shadow p-6 sm:p-8 space-y-6 mb-6">
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800 pb-4 border-b-2 border-purple-200">
                <i class="fas fa-credit-card text-purple-600 mr-3"></i><?php echo trans('payment_gateway') ?? 'Payment Gateway'; ?>
            </h2>
            <div class="space-y-4">
                <!-- Campay API Token -->
                <div>
                    <label class="block text-gray-700 font-semibold mb-2">
                        <i class="fas fa-mobile-alt text-purple-600 mr-2"></i><?php echo trans('campay_api_token') ?? 'Campay API Token (JWT)'; ?>
                    </label>
                    <textarea name="campay_api_token" id="campayApiToken" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-purple-500 focus:ring-2 focus:ring-purple-200 transition font-mono text-sm" 
                              rows="4" placeholder="Paste your Campay JWT token here..."><?php echo htmlspecialchars($settings['campay_api_token'] ?? ''); ?></textarea>
                    <p class="text-xs text-gray-500 mt-2">
                        <i class="fas fa-info-circle mr-1"></i>
                        <?php echo trans('campay_token_description') ?? 'Get your API token from '; ?><a href="https://campay.net" target="_blank" class="text-purple-600 hover:text-purple-700 underline">campay.net</a>
                    </p>
                </div>

                <!-- Campay Fee Information -->
                <div class="bg-purple-50 border border-purple-200 rounded-lg p-4">
                    <p class="text-sm text-purple-800 font-semibold mb-2">
                        <i class="fas fa-info-circle text-purple-600 mr-2"></i><?php echo trans('campay_fees') ?? 'Campay Charges'; ?>
                    </p>
                    <p class="text-sm text-purple-700">
                        <?php echo trans('campay_fee_description') ?? '3% transaction fee (borne by client). This is automatically added to payment amounts.'; ?>
                    </p>
                </div>

                <!-- Mobile Money Methods Supported -->
                <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <p class="text-sm text-blue-800 font-semibold mb-2">
                        <i class="fas fa-mobile-alt text-blue-600 mr-2"></i><?php echo trans('supported_operators') ?? 'Supported Operators'; ?>
                    </p>
                    <p class="text-sm text-blue-700">
                        <span class="inline-block bg-blue-200 text-blue-900 px-3 py-1 rounded-full font-semibold mr-2">MTN</span>
                        <span class="inline-block bg-blue-200 text-blue-900 px-3 py-1 rounded-full font-semibold">ORANGE</span>
                    </p>
                </div>

                <!-- Save Button -->
                <div class="flex gap-3 pt-4 border-t border-gray-200">
                    <button type="submit" class="bg-purple-600 text-white px-6 py-3 rounded-lg hover:bg-purple-700 transition font-semibold flex items-center">
                        <i class="fas fa-save mr-2"></i><?php echo trans('save_changes') ?? 'Save Changes'; ?>
                    </button>
                </div>
            </div>
        </form>

        <!-- System Information -->
        <div class="bg-white rounded-lg shadow p-6 sm:p-8 mb-6">
            <h3 class="text-lg font-semibold text-gray-700 mb-4 flex items-center">
                <i class="fas fa-info-circle text-blue-600 mr-2"></i><?php echo trans('system_information') ?? 'System Information'; ?>
            </h3>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <p class="text-gray-600 text-sm"><?php echo trans('php_version') ?? 'PHP Version'; ?></p>
                    <p class="text-gray-900 font-semibold"><?php echo phpversion(); ?></p>
                </div>
                <div>
                    <p class="text-gray-600 text-sm"><?php echo trans('server') ?? 'Server'; ?></p>
                    <p class="text-gray-900 font-semibold text-sm"><?php echo htmlspecialchars($_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'); ?></p>
                </div>
                <div>
                    <p class="text-gray-600 text-sm"><?php echo trans('database') ?? 'Database'; ?></p>
                    <p class="text-gray-900 font-semibold">MySQL</p>
                </div>
                <div>
                    <p class="text-gray-600 text-sm"><?php echo trans('current_time') ?? 'Current Time'; ?></p>
                    <p class="text-gray-900 font-semibold"><?php echo date('Y-m-d H:i:s'); ?></p>
                </div>
            </div>
        </div>

        <!-- Backup & Maintenance -->
        <div class="bg-white rounded-lg shadow p-6 sm:p-8">
            <h3 class="text-lg font-semibold text-gray-700 mb-4 flex items-center">
                <i class="fas fa-database text-blue-600 mr-2"></i><?php echo trans('backup') ?? 'Backup & Maintenance'; ?>
            </h3>
            <div class="flex flex-col sm:flex-row gap-3">
                <button onclick="backupDatabase()" type="button" class="bg-green-600 text-white px-6 py-3 rounded-lg hover:bg-green-700 transition font-semibold flex items-center justify-center">
                    <i class="fas fa-download mr-2"></i><?php echo trans('backup_database') ?? 'Backup Database'; ?>
                </button>
                <button onclick="clearCache()" type="button" class="bg-yellow-600 text-white px-6 py-3 rounded-lg hover:bg-yellow-700 transition font-semibold flex items-center justify-center">
                    <i class="fas fa-trash mr-2"></i><?php echo trans('clear_cache') ?? 'Clear Cache'; ?>
                </button>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../../layouts/footer.php'; ?>
