<?php
/**
 * Staff Contract PDF Handler
 * Generates professional employment contracts from staff data
 */

header('Content-Type: text/html; charset=utf-8');

try {
    // Get staff ID from query parameter
    $staff_id = isset($_GET['staff_id']) ? trim($_GET['staff_id']) : null;
    
    if (!$staff_id) {
        throw new Exception('Staff ID is required');
    }

    // Database connection
    $conn = new mysqli($db_host, $db_user, $db_pass, $db_name);
    if ($conn->connect_error) {
        throw new Exception('Database connection failed');
    }

    // Fetch staff data
    $query = "SELECT 
        id, 
        user_id, 
        staff_id, 
        position, 
        zone, 
        status,
        contract_date,
        contract_end,
        total_monthly_pay,
        remunerations,
        notes,
        created_at,
        updated_at
    FROM staff 
    WHERE staff_id = ? OR id = ?
    LIMIT 1";

    $stmt = $conn->prepare($query);
    if (!$stmt) {
        throw new Exception('Query preparation failed: ' . $conn->error);
    }

    $stmt->bind_param('si', $staff_id, $staff_id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        throw new Exception('Staff member not found');
    }

    $staff = $result->fetch_assoc();

    // Parse remunerations JSON
    if (!empty($staff['remunerations'])) {
        $staff['remunerations'] = json_decode($staff['remunerations'], true);
    } else {
        $staff['remunerations'] = [];
    }

    // Company information (can be fetched from config or settings)
    $company_name = 'SHEPHERD SECURITY AND CONSULTANCY INTERNATIONAL';
    $company_address = 'Opposite PCSS Buea Town, P.O.Box 554, Buea';
    $company_email = 'info@ssci.cm';
    $company_phone = 'Tel: (237) 676037526 / (237) 688552345';
    $contract_date = date('Y-m-d');

    // Close statements and connection
    $stmt->close();
    $conn->close();

    // Include the contract template
    include __DIR__ . '/staff_contract.php';

} catch (Exception $e) {
    // Error handling
    http_response_code(400);
    echo "<!DOCTYPE html>
    <html>
    <head>
        <title>Contract Error</title>
        <style>
            body { font-family: Arial, sans-serif; padding: 40px; background: #f5f5f5; }
            .error-container { 
                background: #fff; 
                padding: 30px; 
                border-radius: 8px; 
                border-left: 4px solid #ef4444;
                max-width: 600px;
                margin: 0 auto;
            }
            h1 { color: #ef4444; margin-top: 0; }
            p { color: #6b7280; line-height: 1.6; }
            a { color: #3b82f6; text-decoration: none; }
            a:hover { text-decoration: underline; }
        </style>
    </head>
    <body>
        <div class='error-container'>
            <h1>⚠️ Error Generating Contract</h1>
            <p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>
            <p><a href='javascript:history.back()'>← Go Back</a></p>
        </div>
    </body>
    </html>";
}
?>
